<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\Crypt;

class Student extends Model
{
    protected $fillable = [
        'school_id',
        'name',
        'student_id',
        'email',
        'biometric_template',
    ];

    protected $casts = [
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    public function school(): BelongsTo
    {
        return $this->belongsTo(School::class);
    }

    /**
     * Decrypt biometric template when accessed.
     */
    public function getBiometricTemplateAttribute($value)
    {
        if (is_null($value)) {
            return null;
        }

        try {
            return Crypt::decryptString($value);
        } catch (\Exception $e) {
            // If decryption fails, return raw value to avoid breaking existing data
            return $value;
        }
    }

    /**
     * Encrypt biometric template when setting.
     */
    public function setBiometricTemplateAttribute($value)
    {
        if (is_null($value) || $value === '') {
            $this->attributes['biometric_template'] = null;
            return;
        }

        $this->attributes['biometric_template'] = Crypt::encryptString($value);
    }
}
