<?php

namespace App\Console\Commands;

use App\Models\Student;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Crypt;

class EncryptBiometricTemplates extends Command
{
    protected $signature = 'biometric:encrypt';

    protected $description = 'Encrypt all plaintext biometric templates in the database';

    public function handle()
    {
        $students = Student::whereNotNull('biometric_template')->get();

        if ($students->isEmpty()) {
            $this->info('No students with biometric templates found.');
            return 0;
        }

        $encrypted = 0;
        $skipped = 0;

        foreach ($students as $student) {
            $raw = $student->getAttributeFromArray('biometric_template');

            // Check if already encrypted (encrypted strings start with "eyJ" - base64 encoded JSON)
            if ($this->isEncrypted($raw)) {
                $this->line("⏭️  Student #{$student->id} ({$student->name}) - already encrypted, skipping");
                $skipped++;
                continue;
            }

            try {
                // Re-assign to trigger the mutator
                $student->biometric_template = $raw;
                $student->save();
                $this->line("✅ Student #{$student->id} ({$student->name}) - encrypted successfully");
                $encrypted++;
            } catch (\Exception $e) {
                $this->error("❌ Student #{$student->id} ({$student->name}) - failed: {$e->getMessage()}");
            }
        }

        $this->info("\n--- Summary ---");
        $this->info("Encrypted: {$encrypted}");
        $this->info("Skipped: {$skipped}");
        $this->info("Total processed: " . ($encrypted + $skipped));

        return 0;
    }

    /**
     * Check if a value appears to be encrypted.
     * Encrypted strings typically start with "eyJ" (base64 encoded JSON from Laravel Crypt).
     */
    private function isEncrypted($value)
    {
        if (is_null($value) || !is_string($value)) {
            return false;
        }

        return strpos($value, 'eyJ') === 0 || strpos($value, 'base64:') === 0;
    }
}
